/*
mod_image - An Apache module to perform server side JPEG scaling
Copyright (C) 2001  Nathaniel Woods

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public License
as published by the Free Software Foundation; either version 2
of the License, or any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define SHARED_MODULE

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"
#include "ap_alloc.h"

#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "mod_image.h"

#define SCALEDIMAGE_BASEPATH	"C:\\TEMP\\scaled_images\\"

#define snprintf _snprintf

static int parse_args(request_rec *r, double *scale)
{
	char *query_args;

	query_args = r->args;
	if (!query_args)
		return -1;

	*scale = atof(query_args);
	return 0;
}

static int make_scaledimage_filename(char *buf, size_t buflen, const char *original_name, double scale)
{
	char c;
	size_t len;
	char argsuffix[16];

	/* get the right prefix */
	len = strlen(SCALEDIMAGE_BASEPATH);
	if (buflen <= len)
		return -1;
	strcpy(buf, SCALEDIMAGE_BASEPATH);
	buf += len;
	buflen -= len;

	/* convert the string */
	while((c = *(original_name++)) != '\0')
	{
		/* lets be conservative here */
		if (buflen < 4)
			return -1;

		if (ap_isalnum(c) || (c == '.'))
		{
			*(buf++) = c;
			buflen--;
		}
		else if ((c == ' ') || (c == ':'))
		{
			*(buf++) = '+';
			buflen--;
		}
		else if (c == '/')
		{
			*(buf++) = '_';
			buflen--;
		}
		else
		{
			buf += sprintf(buf, "%%%2x", (int) c);
			buflen -= 3;
		}
	}

	/* write the args */
	if (sprintf(argsuffix, "$%g.jpg", scale) >= (int)buflen)
		return -1;
	strcpy(buf, argsuffix);
	return 0;
}

static void imagescaler_init(server_rec *s, pool *p)
{
	struct stat st;
	if (os_stat(SCALEDIMAGE_BASEPATH, &st) != 0)
		mkdir(SCALEDIMAGE_BASEPATH);
}

static int imagescaler_handler(request_rec *r)
{
	struct stat st;
	double scale;
	char filename[MAX_PATH];
	FILE *f;

	/* if we cannot parse the arguments, don't handle the request */
	if (parse_args(r, &scale) < 0)
		return DECLINED;

	/* if we are not going to do anything interesting, don't handle the request */
	if (scale == 1)
		return DECLINED;

	/* sanity check the request */
	if (r->method_number != M_GET)
		return DECLINED;
	if (r->finfo.st_mode == 0) {
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
			"File does not exist: %s", r->filename);
		return NOT_FOUND;
	}

	/* figure out the name of the cache file */
	make_scaledimage_filename(filename, sizeof(filename) / sizeof(filename[0]),
		r->filename, scale);

	/* do we have to generate this file? */
	if ((os_stat(filename, &st) != 0) || (st.st_size == 0))
	{
		int old_priority;
		HANDLE hThread;
		hThread = GetCurrentThread();
		old_priority = GetThreadPriority(hThread);
		SetThreadPriority(hThread, THREAD_PRIORITY_LOWEST);
		scale_image(filename, r->filename, scale);
		SetThreadPriority(hThread, old_priority);
	}

	/* send the file to the client */
	f = ap_pfopen(r->pool, filename, "rb");
    if (!f) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
		    "file permissions deny server access: %s", r->filename);
		return FORBIDDEN;
    }

	r->content_type = "image/jpeg";
	r->status =	HTTP_OK;
	ap_send_http_header(r);

	if (!r->header_only)
		ap_send_fd(f, r);

	ap_pfclose(r->pool, f);
	return OK;
}

static const handler_rec imagescaler_handlers[] =
{
    {"scaled-image", imagescaler_handler},
    {NULL}
};

module MODULE_VAR_EXPORT image_module =
{
    STANDARD_MODULE_STUFF,
    NULL,               /* module initializer */
    NULL,				/* per-directory config creator */
    NULL,				/* dir config merger */
    NULL,				/* server config creator */
    NULL,				/* server config merger */
    NULL,				/* command table */
    imagescaler_handlers,		/* [9] list of handlers */
    NULL,				/* [2] filename-to-URI translation */
    NULL,				/* [5] check/validate user_id */
    NULL,				/* [6] check user_id is valid *here* */
    NULL,				/* [4] check access by host address */
    NULL,				/* [7] MIME type checker/setter */
    NULL,				/* [8] fixups */
    NULL,				/* [10] logger */
    NULL,				/* [3] header parser */
    NULL,				/* process initializer */
    NULL,				/* process exit/cleanup */
    NULL				/* [1] post read_request handling */
};